clear all; close all; clc

% Determine timestep (hourly)
load_timestep = 8760/12; %monthly
simulation_timestep = 8760/365; %Daily resolution

% Dati relativi al carico e allo spreco di calore
availability = 0.15; % [%] disponibilità di produzione di calore
Pprod = 20e6/8760; %[kW] eccesso effettivo di calore residuo
ratio = [0.1 0.2 0.18 0.38 0.55 0.82 0.9 1 0.48 0.35 0.12 0.2]'; % Ratio dei dati
step_waste_heat = Pprod * ratio * availability * 1000; % W

% %load
% LOAD = load('Real_load_VKBtest.txt'); 
% Input = load('input.txt');
% HL=LOAD(:,2)*10^3; %heating load %[W]

%load
LOAD = load('Real_load.txt'); 
Input = load('input.txt');
HL=0.1*LOAD(:,2)*10^3; %heating load %[W]


% Algoritmo di aggregazione del carico --> Trasformare le informazioni sul carico orario in informazioni mensili sul carico
months_matrix = reshape(HL,[],8760/load_timestep)';
step_DH_load = mean(months_matrix,2); %[W] Carico dal distretto in forma di step
step_DH_load(1:3) = 0;
step_DH_load(end-1:end) = 0;

% Configurazione dei boreholes
Ny = 7; % Numero di boreholes nella direzione Y
Nx = 7; % Numero di boreholes nella direzione X
B = 7.5; % Distanza tra i boreholes
H = 150; %[m] Profondità del borehole
D = 4; %[m] Profondità di sepoltura
rb = 0.055; %[m] Raggio del borehole
Ntot = Ny * Nx; % Numero totale di boreholes
shank = 70e-3;
kfill = 0.6; % Conducibilità termica del materiale di riempimento
pipe_do = 32e-3; %[m] Diametro esterno del tubo a U
ro = pipe_do / 2;
tt = 3e-3; %[m] Spessore del tubo a U
pipe_di = pipe_do - 2 * tt; %[m] Diametro interno del tubo a U
ri = pipe_di / 2;
pipe_k = 0.42; %[W/mK]

% Terreno
kg = 3; % Conducibilità termica del terreno
ag = 1 * 10^-6; %[m^2/s] Diffusività termica del terreno

% Fluido del trasportatore di calore
cpw = 4186; %[J/kgK]
rhow = 1000; %[kg/m^3]
req = (0.414 * ro) + (0.5 * shank);
Rbeff = abs(1 / (2 * pi * kfill) * log((rb / req) * sqrt(req / shank / 2)));

% Modello di pompa di calore
Condenser_Coefficients = load('Condenser_capacity.txt');
Evaporator_Coefficients = load('Evaporator_capacity.txt');
Compressor_Coefficients = load('Compressor_capacity.txt');
CondenserTemperature_Coefficients = load('Condenser_temperature.txt');

index = 6; % Indice del modello di compressore scelto
q = 1 * Evaporator_Coefficients(index,:); % Coefficienti di funzione per il modello evaporatore
C = 1 * Condenser_Coefficients(index,:); % Coefficienti di funzione per il modello condensatore
P = 1 * Compressor_Coefficients(index,:); % Coefficienti di funzione per il modello compressore
g = 1 * CondenserTemperature_Coefficients(index,:); % Coefficienti di funzione per la temperatura di condensazione

% Modello di pompa di calore
To = 20;
Td = 65; % Temperatura di scarico del compressore
Qcl_0 = q(1) + q(2) * To + q(3) * To.^2 + q(4) * To.^3 + q(5) * Td.^3 + q(6) * Td + q(7) * Td .* To + q(8) * Td .* To.^2; % Funzione di capacità evaporatore
Qcd_0 = C(1) + C(2) * Td + C(3) * To + C(4) * Qcl_0 + C(5) * Td .* To + C(6) * Qcl_0 .* Td + C(7) * To.^2 + C(8) * Qcl_0 .* To; % Funzione di capacità condensatore
Pcom_0 = P(1) + P(2) * To + P(3) * Td + P(4) * Qcl_0 .* Td + P(5) * To.^2 + P(6) * Td .* Qcl_0.^2 + P(7) * Qcl_0 + P(8) * To .* Qcl_0.^2; % Funzione di potenza compressore
Tc = g(1) * Td + g(2) * To + g(3) * To.^2 + g(4) + g(5) * Td .* To + g(6) * To.^3 + g(7) * Td.^2 + g(8) * Td .* To.^2; % Funzione di temperatura di condensazione
COP = Qcd_0 / Qcl_0;

% Inizializzazione
n = 8760 / simulation_timestep;
T_gr = 8;
Tb1 = zeros(n+1, 1);
Tf1 = zeros(n+1, 1);

% Configurazione dei boreholes
x = 0:B:(Nx-1)*B;
y = 0:B:(Ny-1)*B;
[X, Y] = meshgrid(x, y);
C1 = [];
C2 = [];
d = 0;

% Modello del campo
for i = 1:Ny
    for j = 1:Nx
        for z = 1:Ny
            for k = 1:Nx
                if i == z && j == k
                    d = rb^2;
                else
                    d = (X(i, j) - X(z, k))^2 + (Y(i, j) - Y(z, k))^2;
                end
                index = find(C2 == d);
                if isempty(index)
                    C2 = [C2, d];
                    C1 = [C1, 1];
                else
                    C1(index) =  C1(index) + 1;
                end
            end
        end
    end
end

ierf = @(x) (x .* erf(x)) - (1 / sqrt(pi) * (1 - exp(-x.^2)));
hc = @(s) H*s; 
dc = @(s) D*s; 
Ie = @(s) sum(C1 .* exp(-C2 .* s.^2)) / (Nx * Ny);
ILS = @(hc, dc) 2 * ierf(hc) + 2 * ierf(hc + (2 * dc)) - ierf((2 * hc) + (2 * dc)) - ierf(2 * dc);
intfunc = @(s) Ie(s) .* ILS(hc(s), dc(s)) / (H.*(s.^2));

% Operazione di stoccaggio
years = 5; % Numero di anni
Tb = cell(years, 1); % Inizializza celle per contenere temperature del borehole
Tf = cell(years, 1); % Inizializza celle per contenere temperature del fluido

% Ciclo per ogni anno
for year = 1:years
    % Prepara vettori per memorizzare le temperature
    Tb{year} = zeros(n, 1);
    Tf{year} = zeros(n, 1);
    
    % Calcolo del flusso termico
    surplus = step_waste_heat - step_DH_load; 
    q = surplus / Ntot / H; % flusso per lunghezza
    q_negative = q < 0;
    q(q_negative) = q(q_negative) * (1 - 1 / COP); 
    qp = [q(1); diff(q)]; 
    tp = linspace(0, 8760, (8760/load_timestep)+1);
    
    % Ciclo per ogni passo temporale
    for i = simulation_timestep:simulation_timestep:8760
        is = i / simulation_timestep;
        deltaT = 0;
        index = find(tp >= i, 1) - 1;
            
        t = i * 3600;
        for j = 1:index
            flux = qp(j);
            t_step = (i - tp(j)) * 3600;
            gfunc = (flux / (4 * pi * kg) * integral(intfunc, 1 / sqrt(4 * ag * t_step), inf, 'ArrayValued', true));
            deltaT = deltaT + gfunc;
        end
    
        % Aggiorna le temperature del borehole e del fluido
        Tb{year}(is) = T_gr + deltaT;
        Tf{year}(is) = Tb{year}(is) + Rbeff * q(index);
    end  

        % Aggiungi la temperatura iniziale
        Tb{year} = [T_gr; Tb{year}];
        Tf{year} = [T_gr; Tf{year}];
        T_gr=Tb{year}(end);
end

time = 0:simulation_timestep:8760; % Vettore temporale


figure;
combined_time = []; % Inizializza un vettore per contenere il tempo combinato
combined_Tb = []; % Inizializza un vettore per contenere la temperatura del borehole combinata
combined_Tf = [];
for year = 1:years
    % Concatena i vettori di tempo e temperatura del borehole
    combined_time = [combined_time, time + (year - 1) * 8760]; % Aggiungi un offset di tempo per ogni anno
    combined_Tb = [combined_Tb; Tb{year}];
    combined_Tf = [combined_Tf; Tf{year}];
end

figure(1)
plot(combined_time/8760, combined_Tb, 'LineWidth', 1.5);
hold on
plot(combined_time/8760, combined_Tf, 'LineWidth', 1.5);
xlabel('Time [years]');
ylabel('Temperature [°C]');
legend('Wall temperature','Average fluid temperature','location','best')
set(gca,'fontsize',30)
